<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Fields.text
 *
 * @copyright   (C) 2017 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\Fields\File\Extension;

use Joomla\Component\Fields\Administrator\Plugin\FieldsPlugin;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Component\ComponentHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Fields File Plugin
 *
 * @since  3.7.0
 */
final class File extends FieldsPlugin implements SubscriberInterface
{
    private $fields;
    
    public static function getSubscribedEvents(): array
    {
        return array_merge(parent::getSubscribedEvents(), [
            'onContentAfterSave' => 'afterSave',
            'onCustomFieldsBeforePrepareField' => 'beforePrepareField',
        ]);
    }

    public function __construct($name, array $arguments = [])
    {
        //~ jbdump($arguments);
        $this->fields = [];
        parent::__construct($name, $arguments);
    }


    /**
     * Transforms the field into a DOM XML element and appends it as a child on the given parent.
     *
     * @param   \stdClass    $field   The field.
     * @param   \DOMElement  $parent  The field node parent.
     * @param   Form         $form    The form.
     *
     * @return  ?\DOMElement
     *
     * @since   4.0.0
     */
    public function onCustomFieldsPrepareDom($field, \DOMElement $parent, \Joomla\CMS\Form\Form $form)
    {
        $fieldNode = parent::onCustomFieldsPrepareDom($field, $parent, $form);

        if (!$fieldNode) {
            return $fieldNode;
        }

        $this->fields[] = $field;

        $doc = Factory::getDocument();
        $js = <<<JS
            window.addEventListener('DOMContentLoaded', () => {
                document.adminForm.enctype = 'multipart/form-data';
            });
JS;
        $doc->addScriptDeclaration($js);

        return $fieldNode;
    }

    public function afterSave($args){
        $app = $this->getApplication();
        $post = $app->input->files->get('jform');
        $itemid = intval($args->getItem()->id);

        foreach($this->fields as $field){
            if(empty($field) || empty($field->fieldparams)) continue;
            if(!isset($post['com_fields']) || !isset($post['com_fields'][$field->name])) continue;

            //~ jbdump($args->getData());

            //~ file_put_contents(__FILE__ .'.log', print_r(['post' => $post, 'fieldparams' => $this->field->fieldparams->toArray(), 'field' => $this->field],1));

            $directory = $field->fieldparams->get('directory', '');

            $mediaParams = ComponentHelper::getParams('com_media');

            $paths = [
                'local-files:' => $mediaParams->get('file_path', 'images'),
                'local-images:' => $mediaParams->get('image_path', 'images'),
            ];

            $directory = strtr($directory, $paths);

            if(!file_exists(JPATH_SITE.'/'.$directory.'/')) return;

            $directory .= '/'.$itemid;
            if(!file_exists(JPATH_SITE.'/'.$directory.'/')) mkdir(JPATH_SITE.'/'.$directory.'/', 0755, true);
            
            $multiple = $field->fieldparams->get('multiple', '');

            $value = $post['com_fields'][$field->name];
            if(!$multiple) $value = [$value];

            $vals = [];

            foreach($value as $v => $val){
                if($val['error']) continue;
                $dst = JPATH_SITE.'/'.$directory.'/'.$val['name'];
                if(file_exists($dst)) unlink($dst);
                copy($val['tmp_name'], $dst);
                if(file_exists($dst)) $vals[$v] = $directory.'/'.$val['name'];
            }

            $has = glob(JPATH_SITE.'/'.$directory.'/*.*');
            $has = str_replace(JPATH_SITE.'/', '', $has);

            if(empty($vals)){
                $vals = $has;
            }else{
                foreach($has as $f){
                    if(!in_array($f, $vals)) unlink(JPATH_SITE.'/'.$f);
                }
            }

            if(!empty($vals)){
                $db = Factory::getDbo();
                //~ $db->setQuery("DELETE FROM `#__fields_values` WHERE `field_id` = ".intval($field->id)." AND `item_id` = ".intval($itemid))->execute();
                $fv = new \stdClass;
                $fv->field_id = intval($field->id);
                $fv->item_id = intval($itemid);
                $fv->value = !$multiple ? $vals[0] : json_encode($vals);
                $db->insertObject("#__fields_values", $fv);
            }
        }
    }


    /**
     * Before prepares the field value.
     *
     * @param   BeforePrepareFieldEvent $event    The event instance.
     *
     * @return  void
     *
     * @since   4.0.0
     */
    public function beforePrepareField($event): void
    {
        $field = $event->getField();

        // Check if the field should be processed by us
        if (!$this->isTypeSupported($field->type)) {
            return;
        }

        $multiple = (int)$field->fieldparams->get('multiple', 0);
        if($multiple) $field->value = json_decode($field->value, true);

        // Check if the field value is an old (string) value
        //~ $field->value = $this->checkValue($field->value);
    }
}
